#Requires -RunAsAdministrator

function SafeSaveShortcut($FullName, $Shortcut) {
	$WasReadonly = (Get-ChildItem -Path $FullName).IsReadOnly

	try {
		if ($WasReadOnly) {
			Set-ItemProperty -Path $FullName -Name IsReadOnly -Value $false
		}

		$Shortcut.Save()

		Write-Host "Updated the shortcut $FullName"
	}
	catch {
		Write-Host "Failed to rewrite shortcut: $_"
	}
	finally {
		if ($WasReadOnly -And (-Not (Get-ChildItem $FullName).IsReadOnly)) {
			Set-ItemProperty -Path $FullName -Name IsReadOnly -Value $true
		}
	}
}

function UpdateShortcuts($TaskBarPath, $RdmExePath, $IsRdmLauncher) {
	$WScriptShell = New-Object -ComObject WScript.Shell
	$UpdatedShortcuts = $false

	Write-Host "Checking for taskbar pins in $TaskBarPath..."

	Get-ChildItem $TaskBarPath -Filter '*.lnk' | ForEach-Object {
		try
		{
			$Shortcut = $WScriptShell.CreateShortcut($_.FullName)

			if (-Not $IsRdmLauncher) {
				if ($Shortcut.TargetPath -Like "*RemoteDesktopManager*.exe") {
					$Shortcut.TargetPath = $RdmExePath.FullName
					$Shortcut.IconLocation = "$($RdmExePath.FullName),0"
					SafeSaveShortcut $_.FullName $Shortcut
					$UpdatedShortcuts = $true
				}
			} else {
				if ($Shortcut.TargetPath -Like "*Devolutions*Launcher.exe") {
					$Shortcut.IconLocation = "$($RdmExePath.FullName),0"
					SafeSaveShortcut $_.FullName $Shortcut
					$UpdatedShortcuts = $true
				}
			}
		}
		catch
		{
			Write-Host "Failed to rewrite shortcut: $_"
		}
	}

	return $UpdatedShortcuts
}

function RestoreTombstonedShortcuts($TombstonePath, $IsRdmLauncher) {
	$WScriptShell = New-Object -ComObject WScript.Shell
	$RestoredShortcuts = $false

	Write-Host "Restoring tombstoned taskbar pins..."

	Get-ChildItem $TaskBarTombstonePath -Filter '*.lnk' | ForEach-Object {
		try
		{
			$Shortcut = $WScriptShell.CreateShortcut($_.FullName)
			$RestoreShortcut = (($IsRdmLauncher -And ($Shortcut.TargetPath -Like "*Devolutions*Launcher.exe")) -Or ($Shortcut.TargetPath -Like "*RemoteDesktopManager*.exe"))
			
			if ($RestoreShortcut) {
				Move-Item -Path $_.FullName -Destination $_.Directory.Parent.FullName
				$RestoredShortcuts = $true
			}
		}
		catch
		{
			Write-Host "Failed to restore tombstoned shortcut: $_"
		}
	}

	return $RestoredShortcuts
}

$RdmBasePath = if ($PSScriptRoot) { $PSScriptRoot } else { Get-Location }
Write-Output "RDM Base Path: '$RdmBasePath'"

$RdmExePath = Get-ChildItem -Path "$RdmBasePath\RemoteDesktopManager.exe" | Select-Object -First 1

if (-Not $RdmExePath) {
	$RdmExePath = Get-Item -Path "$RdmBasePath\Launcher.exe"
	$IsRdmLauncher = $true
}

if (-Not (Test-Path $RdmExePath)) {
	throw "Could not find RDM executable in '$RdmBasePath'"
}

Write-Output "RDM Executable: '$RdmExePath'"

$Arm64 = $false

$Wow64Helper = @"
using System;
using System.Runtime.InteropServices;

public static class Wow64Helper {
    [DllImport("kernel32.dll", SetLastError = true)]
    public static extern bool IsWow64Process2(
        IntPtr hProcess, 
        out ushort pProcessMachine, 
        out ushort pNativeMachine
    );
}
"@

try
{
	Add-Type -TypeDefinition $Wow64Helper
	$Process = Get-Process -Id $PID
	$Handle = $Process.handle
	[System.UInt16]$ProcessMachine = 0
	[System.UInt16]$NativeMachine = 0
	$Result = [Wow64Helper]::IsWow64Process2($handle, [ref]$ProcessMachine, [ref]$NativeMachine)

	if ($Result) {
		Write-Host ("Native Machine: 0x{0:X}" -f $NativeMachine)
		$Arm64 = ($NativeMachine -Eq 0xAA64)
	} else {
		$ErrorCode = [System.Runtime.InteropServices.Marshal]::GetLastWin32Error()
		Write-Host "IsWow64Process2 failed with error code $ErrorCode"
	}
}
catch
{
	Write-Host "Failed to check architecture with Win32: $_"

	if ([System.Environment]::Is64BitOperatingSystem)
	{
		if (($Env:PROCESSOR_ARCHITECTURE -eq 'ARM64') -or ($Env:PROCESSOR_ARCHITEW6432 -eq 'ARM64'))
		{
			$Arm64 = $true
		}
	}
}

if ($Arm64)
{
	Write-Host "Enabling Windows ARM64 native execution"

	$Executables = @($RdmExePath)
	foreach ($Executable in $Executables)
	{
		$LeafName = [IO.Path]::GetFileNameWithoutExtension($Executable)
		$LeafName += "_ARM64.exe"
		$ArchExecutable = Join-Path -Path $RdmBasePath -ChildPath $LeafName

		if (-Not (Test-Path $ArchExecutable)) {
			throw "Could not find executable in '$ArchExecutable'"
		}

		Write-Output "Replacing '$Executable' with '$ArchExecutable'..."
		Copy-Item -Path $ArchExecutable $Executable -Force
	}
}

Write-Host "Updating taskbar pinned shortcuts"

Get-ChildItem -Path 'HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\ProfileList' |
	Where-Object { $_.PSChildName.StartsWith('S-1-5-21-') } | ForEach-Object {
		try
		{
			$ProfileImagePath = $_ | Get-ItemPropertyValue -Name 'ProfileImagePath'
			$TaskBarPath = Join-Path $ProfileImagePath "AppData\Roaming\Microsoft\Internet Explorer\Quick Launch\User Pinned\TaskBar"
			$TaskBarTombstonePath = Join-Path $TaskBarPath "Tombstones"
			
			$UpdatedShortcuts = UpdateShortcuts $TaskBarPath $RdmExePath $IsRdmLauncher

			if (-Not $UpdatedShortcuts -And (Test-Path -Path $TaskBarTombstonePath)) {
				$UpdatedShortcuts = RestoreTombstonedShortcuts $TombstonePath $IsRdmLauncher

				if ($UpdatedShortcuts) {
					UpdateShortcuts $TaskBarPath $RdmExePath $IsRdmLauncher
				}
			}
		}
		catch 
		{
			Write-Host "Failed to get ProfileImagePath: $_"
		}
	}

$SHChangeNotify = @'
  [System.Runtime.InteropServices.DllImport("Shell32.dll")] 
  private static extern int SHChangeNotify(int eventId, int flags, IntPtr item1, IntPtr item2);

  public static void Refresh()  {
      SHChangeNotify(0x8000000, 0x1000, IntPtr.Zero, IntPtr.Zero);
  }
'@

try
{
	Add-Type -MemberDefinition $SHChangeNotify -Namespace WinAPI -Name Explorer
	[WinAPI.Explorer]::Refresh()
}
catch
{
	Write-Host "Failed to refresh icon cache: $_"
}

# SIG # Begin signature block
# MIIvWQYJKoZIhvcNAQcCoIIvSjCCL0YCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDjiPTQlbKFbQBs
# 4eyH7ekKXmNp1mUiot0sR1chnQJNnaCCFBcwggVyMIIDWqADAgECAhB2U/6sdUZI
# k/Xl10pIOk74MA0GCSqGSIb3DQEBDAUAMFMxCzAJBgNVBAYTAkJFMRkwFwYDVQQK
# ExBHbG9iYWxTaWduIG52LXNhMSkwJwYDVQQDEyBHbG9iYWxTaWduIENvZGUgU2ln
# bmluZyBSb290IFI0NTAeFw0yMDAzMTgwMDAwMDBaFw00NTAzMTgwMDAwMDBaMFMx
# CzAJBgNVBAYTAkJFMRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNhMSkwJwYDVQQD
# EyBHbG9iYWxTaWduIENvZGUgU2lnbmluZyBSb290IFI0NTCCAiIwDQYJKoZIhvcN
# AQEBBQADggIPADCCAgoCggIBALYtxTDdeuirkD0DcrA6S5kWYbLl/6VnHTcc5X7s
# k4OqhPWjQ5uYRYq4Y1ddmwCIBCXp+GiSS4LYS8lKA/Oof2qPimEnvaFE0P31PyLC
# o0+RjbMFsiiCkV37WYgFC5cGwpj4LKczJO5QOkHM8KCwex1N0qhYOJbp3/kbkbuL
# ECzSx0Mdogl0oYCve+YzCgxZa4689Ktal3t/rlX7hPCA/oRM1+K6vcR1oW+9YRB0
# RLKYB+J0q/9o3GwmPukf5eAEh60w0wyNA3xVuBZwXCR4ICXrZ2eIq7pONJhrcBHe
# OMrUvqHAnOHfHgIB2DvhZ0OEts/8dLcvhKO/ugk3PWdssUVcGWGrQYP1rB3rdw1G
# R3POv72Vle2dK4gQ/vpY6KdX4bPPqFrpByWbEsSegHI9k9yMlN87ROYmgPzSwwPw
# jAzSRdYu54+YnuYE7kJuZ35CFnFi5wT5YMZkobacgSFOK8ZtaJSGxpl0c2cxepHy
# 1Ix5bnymu35Gb03FhRIrz5oiRAiohTfOB2FXBhcSJMDEMXOhmDVXR34QOkXZLaRR
# kJipoAc3xGUaqhxrFnf3p5fsPxkwmW8x++pAsufSxPrJ0PBQdnRZ+o1tFzK++Ol+
# A/Tnh3Wa1EqRLIUDEwIrQoDyiWo2z8hMoM6e+MuNrRan097VmxinxpI68YJj8S4O
# JGTfAgMBAAGjQjBAMA4GA1UdDwEB/wQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB0G
# A1UdDgQWBBQfAL9GgAr8eDm3pbRD2VZQu86WOzANBgkqhkiG9w0BAQwFAAOCAgEA
# Xiu6dJc0RF92SChAhJPuAW7pobPWgCXme+S8CZE9D/x2rdfUMCC7j2DQkdYc8pzv
# eBorlDICwSSWUlIC0PPR/PKbOW6Z4R+OQ0F9mh5byV2ahPwm5ofzdHImraQb2T07
# alKgPAkeLx57szO0Rcf3rLGvk2Ctdq64shV464Nq6//bRqsk5e4C+pAfWcAvXda3
# XaRcELdyU/hBTsz6eBolSsr+hWJDYcO0N6qB0vTWOg+9jVl+MEfeK2vnIVAzX9Rn
# m9S4Z588J5kD/4VDjnMSyiDN6GHVsWbcF9Y5bQ/bzyM3oYKJThxrP9agzaoHnT5C
# JqrXDO76R78aUn7RdYHTyYpiF21PiKAhoCY+r23ZYjAf6Zgorm6N1Y5McmaTgI0q
# 41XHYGeQQlZcIlEPs9xOOe5N3dkdeBBUO27Ql28DtR6yI3PGErKaZND8lYUkqP/f
# obDckUCu3wkzq7ndkrfxzJF0O2nrZ5cbkL/nx6BvcbtXv7ePWu16QGoWzYCELS/h
# AtQklEOzFfwMKxv9cW/8y7x1Fzpeg9LJsy8b1ZyNf1T+fn7kVqOHp53hWVKUQY9t
# W76GlZr/GnbdQNJRSnC0HzNjI3c/7CceWeQIh+00gkoPP/6gHcH1Z3NFhnj0qinp
# J4fGGdvGExTDOUmHTaCX4GUT9Z13Vunas1jHOvLAzYIwggboMIIE0KADAgECAhB3
# vQ4Ft1kLth1HYVMeP3XtMA0GCSqGSIb3DQEBCwUAMFMxCzAJBgNVBAYTAkJFMRkw
# FwYDVQQKExBHbG9iYWxTaWduIG52LXNhMSkwJwYDVQQDEyBHbG9iYWxTaWduIENv
# ZGUgU2lnbmluZyBSb290IFI0NTAeFw0yMDA3MjgwMDAwMDBaFw0zMDA3MjgwMDAw
# MDBaMFwxCzAJBgNVBAYTAkJFMRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNhMTIw
# MAYDVQQDEylHbG9iYWxTaWduIEdDQyBSNDUgRVYgQ29kZVNpZ25pbmcgQ0EgMjAy
# MDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAMsg75ceuQEyQ6BbqYoj
# /SBerjgSi8os1P9B2BpV1BlTt/2jF+d6OVzA984Ro/ml7QH6tbqT76+T3PjisxlM
# g7BKRFAEeIQQaqTWlpCOgfh8qy+1o1cz0lh7lA5tD6WRJiqzg09ysYp7ZJLQ8LRV
# X5YLEeWatSyyEc8lG31RK5gfSaNf+BOeNbgDAtqkEy+FSu/EL3AOwdTMMxLsvUCV
# 0xHK5s2zBZzIU+tS13hMUQGSgt4T8weOdLqEgJ/SpBUO6K/r94n233Hw0b6nskEz
# IHXMsdXtHQcZxOsmd/KrbReTSam35sOQnMa47MzJe5pexcUkk2NvfhCLYc+YVaMk
# oog28vmfvpMusgafJsAMAVYS4bKKnw4e3JiLLs/a4ok0ph8moKiueG3soYgVPMLq
# 7rfYrWGlr3A2onmO3A1zwPHkLKuU7FgGOTZI1jta6CLOdA6vLPEV2tG0leis1Ult
# 5a/dm2tjIF2OfjuyQ9hiOpTlzbSYszcZJBJyc6sEsAnchebUIgTvQCodLm3HadNu
# twFsDeCXpxbmJouI9wNEhl9iZ0y1pzeoVdwDNoxuz202JvEOj7A9ccDhMqeC5LYy
# AjIwfLWTyCH9PIjmaWP47nXJi8Kr77o6/elev7YR8b7wPcoyPm593g9+m5XEEofn
# GrhO7izB36Fl6CSDySrC/blTAgMBAAGjggGtMIIBqTAOBgNVHQ8BAf8EBAMCAYYw
# EwYDVR0lBAwwCgYIKwYBBQUHAwMwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4E
# FgQUJZ3Q/FkJhmPF7POxEztXHAOSNhEwHwYDVR0jBBgwFoAUHwC/RoAK/Hg5t6W0
# Q9lWULvOljswgZMGCCsGAQUFBwEBBIGGMIGDMDkGCCsGAQUFBzABhi1odHRwOi8v
# b2NzcC5nbG9iYWxzaWduLmNvbS9jb2Rlc2lnbmluZ3Jvb3RyNDUwRgYIKwYBBQUH
# MAKGOmh0dHA6Ly9zZWN1cmUuZ2xvYmFsc2lnbi5jb20vY2FjZXJ0L2NvZGVzaWdu
# aW5ncm9vdHI0NS5jcnQwQQYDVR0fBDowODA2oDSgMoYwaHR0cDovL2NybC5nbG9i
# YWxzaWduLmNvbS9jb2Rlc2lnbmluZ3Jvb3RyNDUuY3JsMFUGA1UdIAROMEwwQQYJ
# KwYBBAGgMgECMDQwMgYIKwYBBQUHAgEWJmh0dHBzOi8vd3d3Lmdsb2JhbHNpZ24u
# Y29tL3JlcG9zaXRvcnkvMAcGBWeBDAEDMA0GCSqGSIb3DQEBCwUAA4ICAQAldaAJ
# yTm6t6E5iS8Yn6vW6x1L6JR8DQdomxyd73G2F2prAk+zP4ZFh8xlm0zjWAYCImbV
# YQLFY4/UovG2XiULd5bpzXFAM4gp7O7zom28TbU+BkvJczPKCBQtPUzosLp1pnQt
# pFg6bBNJ+KUVChSWhbFqaDQlQq+WVvQQ+iR98StywRbha+vmqZjHPlr00Bid/XSX
# hndGKj0jfShziq7vKxuav2xTpxSePIdxwF6OyPvTKpIz6ldNXgdeysEYrIEtGiH6
# bs+XYXvfcXo6ymP31TBENzL+u0OF3Lr8psozGSt3bdvLBfB+X3Uuora/Nao2Y8nO
# ZNm9/Lws80lWAMgSK8YnuzevV+/Ezx4pxPTiLc4qYc9X7fUKQOL1GNYe6ZAvytOH
# X5OKSBoRHeU3hZ8uZmKaXoFOlaxVV0PcU4slfjxhD4oLuvU/pteO9wRWXiG7n9dq
# cYC/lt5yA9jYIivzJxZPOOhRQAyuku++PX33gMZMNleElaeEFUgwDlInCI2Oor0i
# xxnJpsoOqHo222q6YV8RJJWk4o5o7hmpSZle0LQ0vdb5QMcQlzFSOTUpEYck08T7
# qWPLd0jV+mL8JOAEek7Q5G7ezp44UCb0IXFl1wkl1MkHAHq4x/N36MXU4lXQ0x72
# f1LiSY25EXIMiEQmM2YBRN/kMw4h3mKJSAfa9TCCB7EwggWZoAMCAQICDHPTwzYD
# /4u0QiTyXjANBgkqhkiG9w0BAQsFADBcMQswCQYDVQQGEwJCRTEZMBcGA1UEChMQ
# R2xvYmFsU2lnbiBudi1zYTEyMDAGA1UEAxMpR2xvYmFsU2lnbiBHQ0MgUjQ1IEVW
# IENvZGVTaWduaW5nIENBIDIwMjAwHhcNMjMxMDMwMTc1MTE4WhcNMjYxMDMwMTc1
# MTE4WjCB8TEdMBsGA1UEDwwUUHJpdmF0ZSBPcmdhbml6YXRpb24xEzARBgNVBAUT
# CjExNjI1NDQ2ODkxEzARBgsrBgEEAYI3PAIBAxMCQ0ExFzAVBgsrBgEEAYI3PAIB
# AhMGUXVlYmVjMQswCQYDVQQGEwJDQTEPMA0GA1UECBMGUXVlYmVjMRIwEAYDVQQH
# EwlMYXZhbHRyaWUxGDAWBgNVBAoTD0Rldm9sdXRpb25zIEluYzEYMBYGA1UEAxMP
# RGV2b2x1dGlvbnMgSW5jMScwJQYJKoZIhvcNAQkBFhhzZWN1cml0eUBkZXZvbHV0
# aW9ucy5uZXQwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQCfDk6c1eCL
# 9rTvq1D1lq1GmU08ZKyYQJQ7Eb/mRFpRXqpOFiySnf8BysYbZ4y4MnIl7M2Wjc5n
# 1JcXR9BPWmkJLnI7rFTwpq/O5xKUwW20/EYyOuF7TasRq6olljm73dcLjrt5z/a2
# u2gO+vMS8LVY6UXKAGZGIigMoPS92f2MkkKmdEmA5dpwbALUfvH9sy0qknUfQY6d
# slpI8PbjTCx9GY5xqCTMtBQcWB5sBn/I0YAlp5yuOn+2ga4vUcucAZTVseoRI/Js
# n5KWWb0iM9wrbv+DOCzcAtBF+Yj2Kp8wHRWfMCumu4YuYcwTY3hbIuNRoUi8j4nL
# ptjGaz7R3UfAr4b/rH4Vg8/l9ufP61Z7bpSkZbIlnh3Gjy9UJCjw5wguQucnllSb
# NNg5ZBd7v3DRUKwKvzF9TYoOERwGdeY8uS4fnSYP7XuGF9b+coZ/D5guGaebiJJE
# odRJkGdiP5P+6jLO43dzgmB4hmWbuF5wofRYXd1ihFOf4aBH2qzHnFkDvp5zeclM
# lgoLuxJVb4mU36Z84KnJuT7fPThK9RbNEoqPPzd1BYcCcRmVaLCYHw+6AgmVXm3b
# gCsv4zM/DqkycfPX11sBXedYdTJ4tihtFo1eRqfQsXEivN+XYwUIJ/EdfHUmaHU+
# 7eYhgSPVynPm9Fq1mAAC3KqH+6RtIpEmpQIDAQABo4IB2zCCAdcwDgYDVR0PAQH/
# BAQDAgeAMIGfBggrBgEFBQcBAQSBkjCBjzBMBggrBgEFBQcwAoZAaHR0cDovL3Nl
# Y3VyZS5nbG9iYWxzaWduLmNvbS9jYWNlcnQvZ3NnY2NyNDVldmNvZGVzaWduY2Ey
# MDIwLmNydDA/BggrBgEFBQcwAYYzaHR0cDovL29jc3AuZ2xvYmFsc2lnbi5jb20v
# Z3NnY2NyNDVldmNvZGVzaWduY2EyMDIwMFUGA1UdIAROMEwwQQYJKwYBBAGgMgEC
# MDQwMgYIKwYBBQUHAgEWJmh0dHBzOi8vd3d3Lmdsb2JhbHNpZ24uY29tL3JlcG9z
# aXRvcnkvMAcGBWeBDAEDMAkGA1UdEwQCMAAwRwYDVR0fBEAwPjA8oDqgOIY2aHR0
# cDovL2NybC5nbG9iYWxzaWduLmNvbS9nc2djY3I0NWV2Y29kZXNpZ25jYTIwMjAu
# Y3JsMCMGA1UdEQQcMBqBGHNlY3VyaXR5QGRldm9sdXRpb25zLm5ldDATBgNVHSUE
# DDAKBggrBgEFBQcDAzAfBgNVHSMEGDAWgBQlndD8WQmGY8Xs87ETO1ccA5I2ETAd
# BgNVHQ4EFgQU+cpn+IPqWRnE5rHeI+bO8b/X89owDQYJKoZIhvcNAQELBQADggIB
# ABr7ukUZYHuRYKb0JdoVh9Lwngn45m/BBg90jTL5CF6ZP4xYB2kaKN366sfAbvmK
# ThbgfcIvN26NjS1/cFXad5af6s5OzGUic+mAFZOhbpX81GedsAnxl1D4BKJs2+iW
# h/eK2vba/K3J5V2Z7S7YFgHqF0vlmDtNxnBQ8jsI30zrbcuYJowft8WLjfW4hr0S
# dAIk2F4X1CTGhtJVMuPcxyUuvrmknp1g2y99jc5eXA6qp0CiUbFC1R3C1kpZYT4s
# xiu86B3kbY6JqTO2f08tjvpih36UeFCC/ByZBzb1D8FFIaKiErjlDHVMIBCY1XrE
# EDEJpIyMRyobXsIuisyn4TpK8JqRb0C0opDzvE8BlKvqlqmHfafbOUXFH5gz/F9a
# iJAMfHyh4ddUg9nFcF+YKWKp8hpdaIW+5ptlsC2LSS5tztMUXRisUf/zCTeLQ2MA
# Xc7Vl0sc8ZD9Uqb9wm+tmK3ZGvnDKCikwE8YU+y96ogFUybGcEWXUYk3QvuXKeS0
# 9/v6QOwbgY3o5EkrNQyPUugI2HsyWtmLhTdDM/Pnj+O2NDNkPXvGiss2b0O8yUMV
# kh9C0HG4WS3L/ExoM1keN1Yd54FaFhk1zQv3KQaC7MJU8uZrmrIJLPNdEPGKiFfI
# 8CLIV/04jAIrR+A4SDaCpDTz+XDZF7kP42KGybJiSD1qMYIamDCCGpQCAQEwbDBc
# MQswCQYDVQQGEwJCRTEZMBcGA1UEChMQR2xvYmFsU2lnbiBudi1zYTEyMDAGA1UE
# AxMpR2xvYmFsU2lnbiBHQ0MgUjQ1IEVWIENvZGVTaWduaW5nIENBIDIwMjACDHPT
# wzYD/4u0QiTyXjANBglghkgBZQMEAgEFAKCBhDAYBgorBgEEAYI3AgEMMQowCKAC
# gAChAoAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsx
# DjAMBgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCBLRo6pVWRGY0HImQ/H18+I
# QBmCpOPJLOsIsNjeLlvKozANBgkqhkiG9w0BAQEFAASCAgBoJaIBq21dbMD/MZe0
# R7ZSZ01BbZFjtVqCFSDNGZlszecTVDXjFH5lcFnapnYzyQz7XFOjvuzVxejMFDxt
# fIMp/Gd6uQA44UUSGc4osdb1oYYY9SdFX2D1mpdYo8nXVdL9xdmMIdz+pkdUroUg
# S4QE3vmGMi+S6/NQLQ3vWuVln0NWKlliXyF0J3hBCN9iNGNktGtCC4n0WEYXXuiT
# efbaWm/yJ6u5j7BZZ5wSBzkwyypS84SmzuannDXG3KO7VS+QZliUOSeP/alQUdFW
# 3zndXP8NK0oxpb/nAsXqK0lunSF5q+OE8jsu1TNWVQVgziR1eLwGIH9Hm8HpPTeF
# kloc5FXuONyY3MSAVMH8WHixCQ168UaL4BWuUj7elu0eqmkarXTg3hcuoZn7Py+A
# eUqY/MnH4YnUI0zGXCKhuyWGYVmEPdVDxNxuh4ZG9ZVKxytUZfcxHZYo09akrexr
# hfoDnUfOnqCzDbrd239xKJX/Z1Zi4TtUmk6is1r4hidC7uIDHBYHnabMe4z96VmU
# aHRpT2MPf7bspWnSrVSfHo9qM9WUtj/mehKSB4t4oMMH/DxNiFsLC4cSiwqvE60Z
# Jb9MJnpNcnuEDtMh2yX2rz+KSn1aSZpmrpuXhXuLymGOHqRhLQaaa+39puttOBn6
# t3WfOXPhRMXjTXS93mKCWxWtzaGCF3YwghdyBgorBgEEAYI3AwMBMYIXYjCCF14G
# CSqGSIb3DQEHAqCCF08wghdLAgEDMQ8wDQYJYIZIAWUDBAIBBQAwdwYLKoZIhvcN
# AQkQAQSgaARmMGQCAQEGCWCGSAGG/WwHATAxMA0GCWCGSAFlAwQCAQUABCCEnMYp
# 1fLNNm7ARgdBo2wwuu3IFeF7umjHg2ZnFCO1hgIQId20o7u+O7RhsZVlkrXEUxgP
# MjAyNTExMDQxMzM0NTNaoIITOjCCBu0wggTVoAMCAQICEAqA7xhLjfEFgtHEdqeV
# dGgwDQYJKoZIhvcNAQELBQAwaTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lD
# ZXJ0LCBJbmMuMUEwPwYDVQQDEzhEaWdpQ2VydCBUcnVzdGVkIEc0IFRpbWVTdGFt
# cGluZyBSU0E0MDk2IFNIQTI1NiAyMDI1IENBMTAeFw0yNTA2MDQwMDAwMDBaFw0z
# NjA5MDMyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgU0hBMjU2IFJTQTQwOTYgVGltZXN0YW1w
# IFJlc3BvbmRlciAyMDI1IDEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoIC
# AQDQRqwtEsae0OquYFazK1e6b1H/hnAKAd/KN8wZQjBjMqiZ3xTWcfsLwOvRxUwX
# cGx8AUjni6bz52fGTfr6PHRNv6T7zsf1Y/E3IU8kgNkeECqVQ+3bzWYesFtkepEr
# vUSbf+EIYLkrLKd6qJnuzK8Vcn0DvbDMemQFoxQ2Dsw4vEjoT1FpS54dNApZfKY6
# 1HAldytxNM89PZXUP/5wWWURK+IfxiOg8W9lKMqzdIo7VA1R0V3Zp3DjjANwqAf4
# lEkTlCDQ0/fKJLKLkzGBTpx6EYevvOi7XOc4zyh1uSqgr6UnbksIcFJqLbkIXIPb
# cNmA98Oskkkrvt6lPAw/p4oDSRZreiwB7x9ykrjS6GS3NR39iTTFS+ENTqW8m6TH
# uOmHHjQNC3zbJ6nJ6SXiLSvw4Smz8U07hqF+8CTXaETkVWz0dVVZw7knh1WZXOLH
# gDvundrAtuvz0D3T+dYaNcwafsVCGZKUhQPL1naFKBy1p6llN3QgshRta6Eq4B40
# h5avMcpi54wm0i2ePZD5pPIssoszQyF4//3DoK2O65Uck5Wggn8O2klETsJ7u8xE
# ehGifgJYi+6I03UuT1j7FnrqVrOzaQoVJOeeStPeldYRNMmSF3voIgMFtNGh86w3
# ISHNm0IaadCKCkUe2LnwJKa8TIlwCUNVwppwn4D3/Pt5pwIDAQABo4IBlTCCAZEw
# DAYDVR0TAQH/BAIwADAdBgNVHQ4EFgQU5Dv88jHt/f3X85FxYxlQQ89hjOgwHwYD
# VR0jBBgwFoAU729TSunkBnx6yuKQVvYv1Ensy04wDgYDVR0PAQH/BAQDAgeAMBYG
# A1UdJQEB/wQMMAoGCCsGAQUFBwMIMIGVBggrBgEFBQcBAQSBiDCBhTAkBggrBgEF
# BQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMF0GCCsGAQUFBzAChlFodHRw
# Oi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRUaW1lU3Rh
# bXBpbmdSU0E0MDk2U0hBMjU2MjAyNUNBMS5jcnQwXwYDVR0fBFgwVjBUoFKgUIZO
# aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0VGltZVN0
# YW1waW5nUlNBNDA5NlNIQTI1NjIwMjVDQTEuY3JsMCAGA1UdIAQZMBcwCAYGZ4EM
# AQQCMAsGCWCGSAGG/WwHATANBgkqhkiG9w0BAQsFAAOCAgEAZSqt8RwnBLmuYEHs
# 0QhEnmNAciH45PYiT9s1i6UKtW+FERp8FgXRGQ/YAavXzWjZhY+hIfP2JkQ38U+w
# tJPBVBajYfrbIYG+Dui4I4PCvHpQuPqFgqp1PzC/ZRX4pvP/ciZmUnthfAEP1HSh
# TrY+2DE5qjzvZs7JIIgt0GCFD9ktx0LxxtRQ7vllKluHWiKk6FxRPyUPxAAYH2Vy
# 1lNM4kzekd8oEARzFAWgeW3az2xejEWLNN4eKGxDJ8WDl/FQUSntbjZ80FU3i54t
# px5F/0Kr15zW/mJAxZMVBrTE2oi0fcI8VMbtoRAmaaslNXdCG1+lqvP4FbrQ6IwS
# BXkZagHLhFU9HCrG/syTRLLhAezu/3Lr00GrJzPQFnCEH1Y58678IgmfORBPC1JK
# kYaEt2OdDh4GmO0/5cHelAK2/gTlQJINqDr6JfwyYHXSd+V08X1JUPvB4ILfJdmL
# +66Gp3CSBXG6IwXMZUXBhtCyIaehr0XkBoDIGMUG1dUtwq1qmcwbdUfcSYCn+Own
# cVUXf53VJUNOaMWMts0VlRYxe5nK+At+DI96HAlXHAL5SlfYxJ7La54i71McVWRP
# 66bW+yERNpbJCjyCYG2j+bdpxo/1Cy4uPcU3AWVPGrbn5PhDBf3Froguzzhk++am
# i+r3Qrx5bIbY3TVzgiFI7Gq3zWcwgga0MIIEnKADAgECAhANx6xXBf8hmS5AQyIM
# OkmGMA0GCSqGSIb3DQEBCwUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
# Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERp
# Z2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yNTA1MDcwMDAwMDBaFw0zODAxMTQy
# MzU5NTlaMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjFB
# MD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBUaW1lU3RhbXBpbmcgUlNBNDA5
# NiBTSEEyNTYgMjAyNSBDQTEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoIC
# AQC0eDHTCphBcr48RsAcrHXbo0ZodLRRF51NrY0NlLWZloMsVO1DahGPNRcybEKq
# +RuwOnPhof6pvF4uGjwjqNjfEvUi6wuim5bap+0lgloM2zX4kftn5B1IpYzTqpyF
# Q/4Bt0mAxAHeHYNnQxqXmRinvuNgxVBdJkf77S2uPoCj7GH8BLuxBG5AvftBdsOE
# CS1UkxBvMgEdgkFiDNYiOTx4OtiFcMSkqTtF2hfQz3zQSku2Ws3IfDReb6e3mmdg
# lTcaarps0wjUjsZvkgFkriK9tUKJm/s80FiocSk1VYLZlDwFt+cVFBURJg6zMUjZ
# a/zbCclF83bRVFLeGkuAhHiGPMvSGmhgaTzVyhYn4p0+8y9oHRaQT/aofEnS5xLr
# fxnGpTXiUOeSLsJygoLPp66bkDX1ZlAeSpQl92QOMeRxykvq6gbylsXQskBBBnGy
# 3tW/AMOMCZIVNSaz7BX8VtYGqLt9MmeOreGPRdtBx3yGOP+rx3rKWDEJlIqLXvJW
# nY0v5ydPpOjL6s36czwzsucuoKs7Yk/ehb//Wx+5kMqIMRvUBDx6z1ev+7psNOdg
# JMoiwOrUG2ZdSoQbU2rMkpLiQ6bGRinZbI4OLu9BMIFm1UUl9VnePs6BaaeEWvjJ
# SjNm2qA+sdFUeEY0qVjPKOWug/G6X5uAiynM7Bu2ayBjUwIDAQABo4IBXTCCAVkw
# EgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU729TSunkBnx6yuKQVvYv1Ens
# y04wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0PAQH/BAQD
# AgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsGAQUFBwEBBGswaTAkBggrBgEF
# BQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAChjVodHRw
# Oi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNy
# dDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGln
# aUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAEGTAXMAgGBmeBDAEEAjALBglg
# hkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBABfO+xaAHP4HPRF2cTC9vgvItTSm
# f83Qh8WIGjB/T8ObXAZz8OjuhUxjaaFdleMM0lBryPTQM2qEJPe36zwbSI/mS83a
# fsl3YTj+IQhQE7jU/kXjjytJgnn0hvrV6hqWGd3rLAUt6vJy9lMDPjTLxLgXf9r5
# nWMQwr8Myb9rEVKChHyfpzee5kH0F8HABBgr0UdqirZ7bowe9Vj2AIMD8liyrukZ
# 2iA/wdG2th9y1IsA0QF8dTXqvcnTmpfeQh35k5zOCPmSNq1UH410ANVko43+Cdmu
# 4y81hjajV/gxdEkMx1NKU4uHQcKfZxAvBAKqMVuqte69M9J6A47OvgRaPs+2ykgc
# GV00TYr2Lr3ty9qIijanrUR3anzEwlvzZiiyfTPjLbnFRsjsYg39OlV8cipDoq7+
# qNNjqFzeGxcytL5TTLL4ZaoBdqbhOhZ3ZRDUphPvSRmMThi0vw9vODRzW6AxnJll
# 38F0cuJG7uEBYTptMSbhdhGQDpOXgpIUsWTjd6xpR6oaQf/DJbg3s6KCLPAlZ66R
# zIg9sC+NJpud/v4+7RWsWCiKi9EOLLHfMR2ZyJ/+xhCx9yHbxtl5TPau1j/1MIDp
# MPx0LckTetiSuEtQvLsNz3Qbp7wGWqbIiOWCnb5WqxL3/BAPvIXKUjPSxyZsq8Wh
# baM2tszWkPZPubdcMIIFjTCCBHWgAwIBAgIQDpsYjvnQLefv21DiCEAYWjANBgkq
# hkiG9w0BAQwFADBlMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5j
# MRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBB
# c3N1cmVkIElEIFJvb3QgQ0EwHhcNMjIwODAxMDAwMDAwWhcNMzExMTA5MjM1OTU5
# WjBiMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQL
# ExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJv
# b3QgRzQwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC/5pBzaN675F1K
# PDAiMGkz7MKnJS7JIT3yithZwuEppz1Yq3aaza57G4QNxDAf8xukOBbrVsaXbR2r
# snnyyhHS5F/WBTxSD1Ifxp4VpX6+n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQLt+C
# 8weE5nQ7bXHiLQwb7iDVySAdYyktzuxeTsiT+CFhmzTrBcZe7FsavOvJz82sNEBf
# sXpm7nfISKhmV1efVFiODCu3T6cw2Vbuyntd463JT17lNecxy9qTXtyOj4DatpGY
# QJB5w3jHtrHEtWoYOAMQjdjUN6QuBX2I9YI+EJFwq1WCQTLX2wRzKm6RAXwhTNS8
# rhsDdV14Ztk6MUSaM0C/CNdaSaTC5qmgZ92kJ7yhTzm1EVgX9yRcRo9k98FpiHaY
# dj1ZXUJ2h4mXaXpI8OCiEhtmmnTK3kse5w5jrubU75KSOp493ADkRSWJtppEGSt+
# wJS00mFt6zPZxd9LBADMfRyVw4/3IbKyEbe7f/LVjHAsQWCqsWMYRJUadmJ+9oCw
# ++hkpjPRiQfhvbfmQ6QYuKZ3AeEPlAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56rF+N
# P8m800ERElvlEFDrMcXKchYiCd98THU/Y+whX8QgUWtvsauGi0/C1kVfnSD8oR7F
# wI+isX4KJpn15GkvmB0t9dmpsh3lGwIDAQABo4IBOjCCATYwDwYDVR0TAQH/BAUw
# AwEB/zAdBgNVHQ4EFgQU7NfjgtJxXWRM3y5nP+e6mK4cD08wHwYDVR0jBBgwFoAU
# Reuir/SSy4IxLVGLp6chnfNtyA8wDgYDVR0PAQH/BAQDAgGGMHkGCCsGAQUFBwEB
# BG0wazAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsG
# AQUFBzAChjdodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1
# cmVkSURSb290Q0EuY3J0MEUGA1UdHwQ+MDwwOqA4oDaGNGh0dHA6Ly9jcmwzLmRp
# Z2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwEQYDVR0gBAow
# CDAGBgRVHSAAMA0GCSqGSIb3DQEBDAUAA4IBAQBwoL9DXFXnOF+go3QbPbYW1/e/
# Vwe9mqyhhyzshV6pGrsi+IcaaVQi7aSId229GhT0E0p6Ly23OO/0/4C5+KH38nLe
# JLxSA8hO0Cre+i1Wz/n096wwepqLsl7Uz9FDRJtDIeuWcqFItJnLnU+nBgMTdydE
# 1Od/6Fmo8L8vC6bp8jQ87PcDx4eo0kxAGTVGamlUsLihVo7spNU96LHc/RzY9Hda
# XFSMb++hUD38dglohJ9vytsgjTVgHAIDyyCwrFigDkBjxZgiwbJZ9VVrzyerbHbO
# byMt9H5xaiNrIv8SuFQtJ37YOtnwtoeW/VvRXKwYw02fc7cBqZ9Xql4o4rmUMYID
# fDCCA3gCAQEwfTBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIElu
# Yy4xQTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgVGltZVN0YW1waW5nIFJT
# QTQwOTYgU0hBMjU2IDIwMjUgQ0ExAhAKgO8YS43xBYLRxHanlXRoMA0GCWCGSAFl
# AwQCAQUAoIHRMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAcBgkqhkiG9w0B
# CQUxDxcNMjUxMTA0MTMzNDUzWjArBgsqhkiG9w0BCRACDDEcMBowGDAWBBTdYjCs
# hgotMGvaOLFoeVIwB/tBfjAvBgkqhkiG9w0BCQQxIgQg5fK6sXCTGFXz3bIx+3AZ
# 40idrarvir/TifFlHZmA3g4wNwYLKoZIhvcNAQkQAi8xKDAmMCQwIgQgSqA/oizX
# XITFXJOPgo5na5yuyrM/420mmqM08UYRCjMwDQYJKoZIhvcNAQEBBQAEggIABnAC
# 90Bna1n0ZpKiM9wFNm6H5nLoN1yEWIaBc0sQeBnOmPZnTr48fHSgH9kwTmGj4Y0D
# 0RWDxO9B5Y3+sxlxyJXplxS3dN4fvnWPaZq7H+3W31GVF9eLw/E/q2Cv08/nuOO+
# 9iygWiBHZ8CEersulweMtkaMnNl1cf8okQ+TUFHA+K4VpCw3x2ufz9NFAOZTbgHK
# db0KAbXGf3haR4jAOLGbBKMdiZ2Ttp7Fyp/Wz7MbqPG5nfASIzBCFNHBrjaOYBAG
# BhPnjlf6ok3hYONv7Re75rqLOGaIwr5SoeN3yOl8DFak8EblWWRxJb4BSJlC6A8B
# 1XIO5CIu9i2YR91EYfiBQa2OXCgCpRap1eTJA0FUA8f4fXyWMAq+lKLddN0eCrJZ
# Q9KiNw5kzhSM3MapTmmxUkxjBBGDtimxluuZujBmMI5ypbyE2VUj+NXWIUOxRSTa
# SFgjRE2RWu2r7ngvPor3VeI1QlWSLQAdJKK7pHAezJDe5XopENrbbDNd+X4db0Wv
# OsrJAqWFiOaQAgZ84DZLeIMlJykaTK7QBmCFgtr403oULh5kBDu6IVCdtIULx9dZ
# SbAJQDaW06UjHyX7Iu/K+MwyDr33kryzQKqTraXUT0Iry6dDQZo/j6wjCu7aulAZ
# 4isj6pQoLwlbP7XmSeA1uNmjpnHTpirxnoTHaiE=
# SIG # End signature block
