#Requires -RunAsAdministrator

function SafeSaveShortcut($FullName, $Shortcut) {
	$WasReadonly = (Get-ChildItem -Path $FullName).IsReadOnly

	try {
		if ($WasReadOnly) {
			Set-ItemProperty -Path $FullName -Name IsReadOnly -Value $false
		}

		$Shortcut.Save()

		Write-Host "Updated the shortcut $FullName"
	}
	catch {
		Write-Host "Failed to rewrite shortcut: $_"
	}
	finally {
		if ($WasReadOnly -And (-Not (Get-ChildItem $FullName).IsReadOnly)) {
			Set-ItemProperty -Path $FullName -Name IsReadOnly -Value $true
		}
	}
}

function UpdateShortcuts($TaskBarPath, $RdmExePath, $IsRdmLauncher) {
	$WScriptShell = New-Object -ComObject WScript.Shell
	$UpdatedShortcuts = $false

	Write-Host "Checking for taskbar pins in $TaskBarPath..."

	Get-ChildItem $TaskBarPath -Filter '*.lnk' | ForEach-Object {
		try
		{
			$Shortcut = $WScriptShell.CreateShortcut($_.FullName)

			if (-Not $IsRdmLauncher) {
				if ($Shortcut.TargetPath -Like "*RemoteDesktopManager*.exe") {
					$Shortcut.TargetPath = $RdmExePath.FullName
					$Shortcut.IconLocation = "$($RdmExePath.FullName),0"
					SafeSaveShortcut $_.FullName $Shortcut
					$UpdatedShortcuts = $true
				}
			} else {
				if ($Shortcut.TargetPath -Like "*Devolutions*Launcher.exe") {
					$Shortcut.IconLocation = "$($RdmExePath.FullName),0"
					SafeSaveShortcut $_.FullName $Shortcut
					$UpdatedShortcuts = $true
				}
			}
		}
		catch
		{
			Write-Host "Failed to rewrite shortcut: $_"
		}
	}

	return $UpdatedShortcuts
}

function RestoreTombstonedShortcuts($TombstonePath, $IsRdmLauncher) {
	$WScriptShell = New-Object -ComObject WScript.Shell
	$RestoredShortcuts = $false

	Write-Host "Restoring tombstoned taskbar pins..."

	Get-ChildItem $TaskBarTombstonePath -Filter '*.lnk' | ForEach-Object {
		try
		{
			$Shortcut = $WScriptShell.CreateShortcut($_.FullName)
			$RestoreShortcut = (($IsRdmLauncher -And ($Shortcut.TargetPath -Like "*Devolutions*Launcher.exe")) -Or ($Shortcut.TargetPath -Like "*RemoteDesktopManager*.exe"))
			
			if ($RestoreShortcut) {
				Move-Item -Path $_.FullName -Destination $_.Directory.Parent.FullName
				$RestoredShortcuts = $true
			}
		}
		catch
		{
			Write-Host "Failed to restore tombstoned shortcut: $_"
		}
	}

	return $RestoredShortcuts
}

$RdmBasePath = if ($PSScriptRoot) { $PSScriptRoot } else { Get-Location }
Write-Output "RDM Base Path: '$RdmBasePath'"

$RdmExePath = Get-ChildItem -Path "$RdmBasePath\RemoteDesktopManager.exe" | Select-Object -First 1

if (-Not $RdmExePath) {
	$RdmExePath = Get-Item -Path "$RdmBasePath\Launcher.exe"
	$IsRdmLauncher = $true
}

if (-Not (Test-Path $RdmExePath)) {
	throw "Could not find RDM executable in '$RdmBasePath'"
}

Write-Output "RDM Executable: '$RdmExePath'"

$Arm64 = $false

$Wow64Helper = @"
using System;
using System.Runtime.InteropServices;

public static class Wow64Helper {
    [DllImport("kernel32.dll", SetLastError = true)]
    public static extern bool IsWow64Process2(
        IntPtr hProcess, 
        out ushort pProcessMachine, 
        out ushort pNativeMachine
    );
}
"@

try
{
	Add-Type -TypeDefinition $Wow64Helper
	$Process = Get-Process -Id $PID
	$Handle = $Process.handle
	[System.UInt16]$ProcessMachine = 0
	[System.UInt16]$NativeMachine = 0
	$Result = [Wow64Helper]::IsWow64Process2($handle, [ref]$ProcessMachine, [ref]$NativeMachine)

	if ($Result) {
		Write-Host ("Native Machine: 0x{0:X}" -f $NativeMachine)
		$Arm64 = ($NativeMachine -Eq 0xAA64)
	} else {
		$ErrorCode = [System.Runtime.InteropServices.Marshal]::GetLastWin32Error()
		Write-Host "IsWow64Process2 failed with error code $ErrorCode"
	}
}
catch
{
	Write-Host "Failed to check architecture with Win32: $_"

	if ([System.Environment]::Is64BitOperatingSystem)
	{
		if (($Env:PROCESSOR_ARCHITECTURE -eq 'ARM64') -or ($Env:PROCESSOR_ARCHITEW6432 -eq 'ARM64'))
		{
			$Arm64 = $true
		}
	}
}

if ($Arm64)
{
	Write-Host "Enabling Windows ARM64 native execution"

	$Executables = @($RdmExePath)
	foreach ($Executable in $Executables)
	{
		$LeafName = [IO.Path]::GetFileNameWithoutExtension($Executable)
		$LeafName += "_ARM64.exe"
		$ArchExecutable = Join-Path -Path $RdmBasePath -ChildPath $LeafName

		if (-Not (Test-Path $ArchExecutable)) {
			throw "Could not find executable in '$ArchExecutable'"
		}

		Write-Output "Replacing '$Executable' with '$ArchExecutable'..."
		Copy-Item -Path $ArchExecutable $Executable -Force
	}
}

Write-Host "Updating taskbar pinned shortcuts"

Get-ChildItem -Path 'HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\ProfileList' |
	Where-Object { $_.PSChildName.StartsWith('S-1-5-21-') } | ForEach-Object {
		try
		{
			$ProfileImagePath = $_ | Get-ItemPropertyValue -Name 'ProfileImagePath'
			$TaskBarPath = Join-Path $ProfileImagePath "AppData\Roaming\Microsoft\Internet Explorer\Quick Launch\User Pinned\TaskBar"
			$TaskBarTombstonePath = Join-Path $TaskBarPath "Tombstones"
			
			$UpdatedShortcuts = UpdateShortcuts $TaskBarPath $RdmExePath $IsRdmLauncher

			if (-Not $UpdatedShortcuts -And (Test-Path -Path $TaskBarTombstonePath)) {
				$UpdatedShortcuts = RestoreTombstonedShortcuts $TombstonePath $IsRdmLauncher

				if ($UpdatedShortcuts) {
					UpdateShortcuts $TaskBarPath $RdmExePath $IsRdmLauncher
				}
			}
		}
		catch 
		{
			Write-Host "Failed to get ProfileImagePath: $_"
		}
	}

$SHChangeNotify = @'
  [System.Runtime.InteropServices.DllImport("Shell32.dll")] 
  private static extern int SHChangeNotify(int eventId, int flags, IntPtr item1, IntPtr item2);

  public static void Refresh()  {
      SHChangeNotify(0x8000000, 0x1000, IntPtr.Zero, IntPtr.Zero);
  }
'@

try
{
	Add-Type -MemberDefinition $SHChangeNotify -Namespace WinAPI -Name Explorer
	[WinAPI.Explorer]::Refresh()
}
catch
{
	Write-Host "Failed to refresh icon cache: $_"
}

# SIG # Begin signature block
# MIIvHQYJKoZIhvcNAQcCoIIvDjCCLwoCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDjiPTQlbKFbQBs
# 4eyH7ekKXmNp1mUiot0sR1chnQJNnaCCFBcwggVyMIIDWqADAgECAhB2U/6sdUZI
# k/Xl10pIOk74MA0GCSqGSIb3DQEBDAUAMFMxCzAJBgNVBAYTAkJFMRkwFwYDVQQK
# ExBHbG9iYWxTaWduIG52LXNhMSkwJwYDVQQDEyBHbG9iYWxTaWduIENvZGUgU2ln
# bmluZyBSb290IFI0NTAeFw0yMDAzMTgwMDAwMDBaFw00NTAzMTgwMDAwMDBaMFMx
# CzAJBgNVBAYTAkJFMRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNhMSkwJwYDVQQD
# EyBHbG9iYWxTaWduIENvZGUgU2lnbmluZyBSb290IFI0NTCCAiIwDQYJKoZIhvcN
# AQEBBQADggIPADCCAgoCggIBALYtxTDdeuirkD0DcrA6S5kWYbLl/6VnHTcc5X7s
# k4OqhPWjQ5uYRYq4Y1ddmwCIBCXp+GiSS4LYS8lKA/Oof2qPimEnvaFE0P31PyLC
# o0+RjbMFsiiCkV37WYgFC5cGwpj4LKczJO5QOkHM8KCwex1N0qhYOJbp3/kbkbuL
# ECzSx0Mdogl0oYCve+YzCgxZa4689Ktal3t/rlX7hPCA/oRM1+K6vcR1oW+9YRB0
# RLKYB+J0q/9o3GwmPukf5eAEh60w0wyNA3xVuBZwXCR4ICXrZ2eIq7pONJhrcBHe
# OMrUvqHAnOHfHgIB2DvhZ0OEts/8dLcvhKO/ugk3PWdssUVcGWGrQYP1rB3rdw1G
# R3POv72Vle2dK4gQ/vpY6KdX4bPPqFrpByWbEsSegHI9k9yMlN87ROYmgPzSwwPw
# jAzSRdYu54+YnuYE7kJuZ35CFnFi5wT5YMZkobacgSFOK8ZtaJSGxpl0c2cxepHy
# 1Ix5bnymu35Gb03FhRIrz5oiRAiohTfOB2FXBhcSJMDEMXOhmDVXR34QOkXZLaRR
# kJipoAc3xGUaqhxrFnf3p5fsPxkwmW8x++pAsufSxPrJ0PBQdnRZ+o1tFzK++Ol+
# A/Tnh3Wa1EqRLIUDEwIrQoDyiWo2z8hMoM6e+MuNrRan097VmxinxpI68YJj8S4O
# JGTfAgMBAAGjQjBAMA4GA1UdDwEB/wQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB0G
# A1UdDgQWBBQfAL9GgAr8eDm3pbRD2VZQu86WOzANBgkqhkiG9w0BAQwFAAOCAgEA
# Xiu6dJc0RF92SChAhJPuAW7pobPWgCXme+S8CZE9D/x2rdfUMCC7j2DQkdYc8pzv
# eBorlDICwSSWUlIC0PPR/PKbOW6Z4R+OQ0F9mh5byV2ahPwm5ofzdHImraQb2T07
# alKgPAkeLx57szO0Rcf3rLGvk2Ctdq64shV464Nq6//bRqsk5e4C+pAfWcAvXda3
# XaRcELdyU/hBTsz6eBolSsr+hWJDYcO0N6qB0vTWOg+9jVl+MEfeK2vnIVAzX9Rn
# m9S4Z588J5kD/4VDjnMSyiDN6GHVsWbcF9Y5bQ/bzyM3oYKJThxrP9agzaoHnT5C
# JqrXDO76R78aUn7RdYHTyYpiF21PiKAhoCY+r23ZYjAf6Zgorm6N1Y5McmaTgI0q
# 41XHYGeQQlZcIlEPs9xOOe5N3dkdeBBUO27Ql28DtR6yI3PGErKaZND8lYUkqP/f
# obDckUCu3wkzq7ndkrfxzJF0O2nrZ5cbkL/nx6BvcbtXv7ePWu16QGoWzYCELS/h
# AtQklEOzFfwMKxv9cW/8y7x1Fzpeg9LJsy8b1ZyNf1T+fn7kVqOHp53hWVKUQY9t
# W76GlZr/GnbdQNJRSnC0HzNjI3c/7CceWeQIh+00gkoPP/6gHcH1Z3NFhnj0qinp
# J4fGGdvGExTDOUmHTaCX4GUT9Z13Vunas1jHOvLAzYIwggboMIIE0KADAgECAhB3
# vQ4Ft1kLth1HYVMeP3XtMA0GCSqGSIb3DQEBCwUAMFMxCzAJBgNVBAYTAkJFMRkw
# FwYDVQQKExBHbG9iYWxTaWduIG52LXNhMSkwJwYDVQQDEyBHbG9iYWxTaWduIENv
# ZGUgU2lnbmluZyBSb290IFI0NTAeFw0yMDA3MjgwMDAwMDBaFw0zMDA3MjgwMDAw
# MDBaMFwxCzAJBgNVBAYTAkJFMRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNhMTIw
# MAYDVQQDEylHbG9iYWxTaWduIEdDQyBSNDUgRVYgQ29kZVNpZ25pbmcgQ0EgMjAy
# MDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAMsg75ceuQEyQ6BbqYoj
# /SBerjgSi8os1P9B2BpV1BlTt/2jF+d6OVzA984Ro/ml7QH6tbqT76+T3PjisxlM
# g7BKRFAEeIQQaqTWlpCOgfh8qy+1o1cz0lh7lA5tD6WRJiqzg09ysYp7ZJLQ8LRV
# X5YLEeWatSyyEc8lG31RK5gfSaNf+BOeNbgDAtqkEy+FSu/EL3AOwdTMMxLsvUCV
# 0xHK5s2zBZzIU+tS13hMUQGSgt4T8weOdLqEgJ/SpBUO6K/r94n233Hw0b6nskEz
# IHXMsdXtHQcZxOsmd/KrbReTSam35sOQnMa47MzJe5pexcUkk2NvfhCLYc+YVaMk
# oog28vmfvpMusgafJsAMAVYS4bKKnw4e3JiLLs/a4ok0ph8moKiueG3soYgVPMLq
# 7rfYrWGlr3A2onmO3A1zwPHkLKuU7FgGOTZI1jta6CLOdA6vLPEV2tG0leis1Ult
# 5a/dm2tjIF2OfjuyQ9hiOpTlzbSYszcZJBJyc6sEsAnchebUIgTvQCodLm3HadNu
# twFsDeCXpxbmJouI9wNEhl9iZ0y1pzeoVdwDNoxuz202JvEOj7A9ccDhMqeC5LYy
# AjIwfLWTyCH9PIjmaWP47nXJi8Kr77o6/elev7YR8b7wPcoyPm593g9+m5XEEofn
# GrhO7izB36Fl6CSDySrC/blTAgMBAAGjggGtMIIBqTAOBgNVHQ8BAf8EBAMCAYYw
# EwYDVR0lBAwwCgYIKwYBBQUHAwMwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4E
# FgQUJZ3Q/FkJhmPF7POxEztXHAOSNhEwHwYDVR0jBBgwFoAUHwC/RoAK/Hg5t6W0
# Q9lWULvOljswgZMGCCsGAQUFBwEBBIGGMIGDMDkGCCsGAQUFBzABhi1odHRwOi8v
# b2NzcC5nbG9iYWxzaWduLmNvbS9jb2Rlc2lnbmluZ3Jvb3RyNDUwRgYIKwYBBQUH
# MAKGOmh0dHA6Ly9zZWN1cmUuZ2xvYmFsc2lnbi5jb20vY2FjZXJ0L2NvZGVzaWdu
# aW5ncm9vdHI0NS5jcnQwQQYDVR0fBDowODA2oDSgMoYwaHR0cDovL2NybC5nbG9i
# YWxzaWduLmNvbS9jb2Rlc2lnbmluZ3Jvb3RyNDUuY3JsMFUGA1UdIAROMEwwQQYJ
# KwYBBAGgMgECMDQwMgYIKwYBBQUHAgEWJmh0dHBzOi8vd3d3Lmdsb2JhbHNpZ24u
# Y29tL3JlcG9zaXRvcnkvMAcGBWeBDAEDMA0GCSqGSIb3DQEBCwUAA4ICAQAldaAJ
# yTm6t6E5iS8Yn6vW6x1L6JR8DQdomxyd73G2F2prAk+zP4ZFh8xlm0zjWAYCImbV
# YQLFY4/UovG2XiULd5bpzXFAM4gp7O7zom28TbU+BkvJczPKCBQtPUzosLp1pnQt
# pFg6bBNJ+KUVChSWhbFqaDQlQq+WVvQQ+iR98StywRbha+vmqZjHPlr00Bid/XSX
# hndGKj0jfShziq7vKxuav2xTpxSePIdxwF6OyPvTKpIz6ldNXgdeysEYrIEtGiH6
# bs+XYXvfcXo6ymP31TBENzL+u0OF3Lr8psozGSt3bdvLBfB+X3Uuora/Nao2Y8nO
# ZNm9/Lws80lWAMgSK8YnuzevV+/Ezx4pxPTiLc4qYc9X7fUKQOL1GNYe6ZAvytOH
# X5OKSBoRHeU3hZ8uZmKaXoFOlaxVV0PcU4slfjxhD4oLuvU/pteO9wRWXiG7n9dq
# cYC/lt5yA9jYIivzJxZPOOhRQAyuku++PX33gMZMNleElaeEFUgwDlInCI2Oor0i
# xxnJpsoOqHo222q6YV8RJJWk4o5o7hmpSZle0LQ0vdb5QMcQlzFSOTUpEYck08T7
# qWPLd0jV+mL8JOAEek7Q5G7ezp44UCb0IXFl1wkl1MkHAHq4x/N36MXU4lXQ0x72
# f1LiSY25EXIMiEQmM2YBRN/kMw4h3mKJSAfa9TCCB7EwggWZoAMCAQICDHPTwzYD
# /4u0QiTyXjANBgkqhkiG9w0BAQsFADBcMQswCQYDVQQGEwJCRTEZMBcGA1UEChMQ
# R2xvYmFsU2lnbiBudi1zYTEyMDAGA1UEAxMpR2xvYmFsU2lnbiBHQ0MgUjQ1IEVW
# IENvZGVTaWduaW5nIENBIDIwMjAwHhcNMjMxMDMwMTc1MTE4WhcNMjYxMDMwMTc1
# MTE4WjCB8TEdMBsGA1UEDwwUUHJpdmF0ZSBPcmdhbml6YXRpb24xEzARBgNVBAUT
# CjExNjI1NDQ2ODkxEzARBgsrBgEEAYI3PAIBAxMCQ0ExFzAVBgsrBgEEAYI3PAIB
# AhMGUXVlYmVjMQswCQYDVQQGEwJDQTEPMA0GA1UECBMGUXVlYmVjMRIwEAYDVQQH
# EwlMYXZhbHRyaWUxGDAWBgNVBAoTD0Rldm9sdXRpb25zIEluYzEYMBYGA1UEAxMP
# RGV2b2x1dGlvbnMgSW5jMScwJQYJKoZIhvcNAQkBFhhzZWN1cml0eUBkZXZvbHV0
# aW9ucy5uZXQwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQCfDk6c1eCL
# 9rTvq1D1lq1GmU08ZKyYQJQ7Eb/mRFpRXqpOFiySnf8BysYbZ4y4MnIl7M2Wjc5n
# 1JcXR9BPWmkJLnI7rFTwpq/O5xKUwW20/EYyOuF7TasRq6olljm73dcLjrt5z/a2
# u2gO+vMS8LVY6UXKAGZGIigMoPS92f2MkkKmdEmA5dpwbALUfvH9sy0qknUfQY6d
# slpI8PbjTCx9GY5xqCTMtBQcWB5sBn/I0YAlp5yuOn+2ga4vUcucAZTVseoRI/Js
# n5KWWb0iM9wrbv+DOCzcAtBF+Yj2Kp8wHRWfMCumu4YuYcwTY3hbIuNRoUi8j4nL
# ptjGaz7R3UfAr4b/rH4Vg8/l9ufP61Z7bpSkZbIlnh3Gjy9UJCjw5wguQucnllSb
# NNg5ZBd7v3DRUKwKvzF9TYoOERwGdeY8uS4fnSYP7XuGF9b+coZ/D5guGaebiJJE
# odRJkGdiP5P+6jLO43dzgmB4hmWbuF5wofRYXd1ihFOf4aBH2qzHnFkDvp5zeclM
# lgoLuxJVb4mU36Z84KnJuT7fPThK9RbNEoqPPzd1BYcCcRmVaLCYHw+6AgmVXm3b
# gCsv4zM/DqkycfPX11sBXedYdTJ4tihtFo1eRqfQsXEivN+XYwUIJ/EdfHUmaHU+
# 7eYhgSPVynPm9Fq1mAAC3KqH+6RtIpEmpQIDAQABo4IB2zCCAdcwDgYDVR0PAQH/
# BAQDAgeAMIGfBggrBgEFBQcBAQSBkjCBjzBMBggrBgEFBQcwAoZAaHR0cDovL3Nl
# Y3VyZS5nbG9iYWxzaWduLmNvbS9jYWNlcnQvZ3NnY2NyNDVldmNvZGVzaWduY2Ey
# MDIwLmNydDA/BggrBgEFBQcwAYYzaHR0cDovL29jc3AuZ2xvYmFsc2lnbi5jb20v
# Z3NnY2NyNDVldmNvZGVzaWduY2EyMDIwMFUGA1UdIAROMEwwQQYJKwYBBAGgMgEC
# MDQwMgYIKwYBBQUHAgEWJmh0dHBzOi8vd3d3Lmdsb2JhbHNpZ24uY29tL3JlcG9z
# aXRvcnkvMAcGBWeBDAEDMAkGA1UdEwQCMAAwRwYDVR0fBEAwPjA8oDqgOIY2aHR0
# cDovL2NybC5nbG9iYWxzaWduLmNvbS9nc2djY3I0NWV2Y29kZXNpZ25jYTIwMjAu
# Y3JsMCMGA1UdEQQcMBqBGHNlY3VyaXR5QGRldm9sdXRpb25zLm5ldDATBgNVHSUE
# DDAKBggrBgEFBQcDAzAfBgNVHSMEGDAWgBQlndD8WQmGY8Xs87ETO1ccA5I2ETAd
# BgNVHQ4EFgQU+cpn+IPqWRnE5rHeI+bO8b/X89owDQYJKoZIhvcNAQELBQADggIB
# ABr7ukUZYHuRYKb0JdoVh9Lwngn45m/BBg90jTL5CF6ZP4xYB2kaKN366sfAbvmK
# ThbgfcIvN26NjS1/cFXad5af6s5OzGUic+mAFZOhbpX81GedsAnxl1D4BKJs2+iW
# h/eK2vba/K3J5V2Z7S7YFgHqF0vlmDtNxnBQ8jsI30zrbcuYJowft8WLjfW4hr0S
# dAIk2F4X1CTGhtJVMuPcxyUuvrmknp1g2y99jc5eXA6qp0CiUbFC1R3C1kpZYT4s
# xiu86B3kbY6JqTO2f08tjvpih36UeFCC/ByZBzb1D8FFIaKiErjlDHVMIBCY1XrE
# EDEJpIyMRyobXsIuisyn4TpK8JqRb0C0opDzvE8BlKvqlqmHfafbOUXFH5gz/F9a
# iJAMfHyh4ddUg9nFcF+YKWKp8hpdaIW+5ptlsC2LSS5tztMUXRisUf/zCTeLQ2MA
# Xc7Vl0sc8ZD9Uqb9wm+tmK3ZGvnDKCikwE8YU+y96ogFUybGcEWXUYk3QvuXKeS0
# 9/v6QOwbgY3o5EkrNQyPUugI2HsyWtmLhTdDM/Pnj+O2NDNkPXvGiss2b0O8yUMV
# kh9C0HG4WS3L/ExoM1keN1Yd54FaFhk1zQv3KQaC7MJU8uZrmrIJLPNdEPGKiFfI
# 8CLIV/04jAIrR+A4SDaCpDTz+XDZF7kP42KGybJiSD1qMYIaXDCCGlgCAQEwbDBc
# MQswCQYDVQQGEwJCRTEZMBcGA1UEChMQR2xvYmFsU2lnbiBudi1zYTEyMDAGA1UE
# AxMpR2xvYmFsU2lnbiBHQ0MgUjQ1IEVWIENvZGVTaWduaW5nIENBIDIwMjACDHPT
# wzYD/4u0QiTyXjANBglghkgBZQMEAgEFAKCBhDAYBgorBgEEAYI3AgEMMQowCKAC
# gAChAoAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsx
# DjAMBgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCBLRo6pVWRGY0HImQ/H18+I
# QBmCpOPJLOsIsNjeLlvKozANBgkqhkiG9w0BAQEFAASCAgBoJaIBq21dbMD/MZe0
# R7ZSZ01BbZFjtVqCFSDNGZlszecTVDXjFH5lcFnapnYzyQz7XFOjvuzVxejMFDxt
# fIMp/Gd6uQA44UUSGc4osdb1oYYY9SdFX2D1mpdYo8nXVdL9xdmMIdz+pkdUroUg
# S4QE3vmGMi+S6/NQLQ3vWuVln0NWKlliXyF0J3hBCN9iNGNktGtCC4n0WEYXXuiT
# efbaWm/yJ6u5j7BZZ5wSBzkwyypS84SmzuannDXG3KO7VS+QZliUOSeP/alQUdFW
# 3zndXP8NK0oxpb/nAsXqK0lunSF5q+OE8jsu1TNWVQVgziR1eLwGIH9Hm8HpPTeF
# kloc5FXuONyY3MSAVMH8WHixCQ168UaL4BWuUj7elu0eqmkarXTg3hcuoZn7Py+A
# eUqY/MnH4YnUI0zGXCKhuyWGYVmEPdVDxNxuh4ZG9ZVKxytUZfcxHZYo09akrexr
# hfoDnUfOnqCzDbrd239xKJX/Z1Zi4TtUmk6is1r4hidC7uIDHBYHnabMe4z96VmU
# aHRpT2MPf7bspWnSrVSfHo9qM9WUtj/mehKSB4t4oMMH/DxNiFsLC4cSiwqvE60Z
# Jb9MJnpNcnuEDtMh2yX2rz+KSn1aSZpmrpuXhXuLymGOHqRhLQaaa+39puttOBn6
# t3WfOXPhRMXjTXS93mKCWxWtzaGCFzowghc2BgorBgEEAYI3AwMBMYIXJjCCFyIG
# CSqGSIb3DQEHAqCCFxMwghcPAgEDMQ8wDQYJYIZIAWUDBAIBBQAweAYLKoZIhvcN
# AQkQAQSgaQRnMGUCAQEGCWCGSAGG/WwHATAxMA0GCWCGSAFlAwQCAQUABCCEnMYp
# 1fLNNm7ARgdBo2wwuu3IFeF7umjHg2ZnFCO1hgIRANeOm7eujgL1GtYO/Ycu6aYY
# DzIwMjUwNDI4MTQ0NzQ1WqCCEwMwgga8MIIEpKADAgECAhALrma8Wrp/lYfG+ekE
# 4zMEMA0GCSqGSIb3DQEBCwUAMGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdp
# Q2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2
# IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwHhcNMjQwOTI2MDAwMDAwWhcNMzUxMTI1
# MjM1OTU5WjBCMQswCQYDVQQGEwJVUzERMA8GA1UEChMIRGlnaUNlcnQxIDAeBgNV
# BAMTF0RpZ2lDZXJ0IFRpbWVzdGFtcCAyMDI0MIICIjANBgkqhkiG9w0BAQEFAAOC
# Ag8AMIICCgKCAgEAvmpzn/aVIauWMLpbbeZZo7Xo/ZEfGMSIO2qZ46XB/QowIEMS
# vgjEdEZ3v4vrrTHleW1JWGErrjOL0J4L0HqVR1czSzvUQ5xF7z4IQmn7dHY7yijv
# oQ7ujm0u6yXF2v1CrzZopykD07/9fpAT4BxpT9vJoJqAsP8YuhRvflJ9YeHjes4f
# duksTHulntq9WelRWY++TFPxzZrbILRYynyEy7rS1lHQKFpXvo2GePfsMRhNf1F4
# 1nyEg5h7iOXv+vjX0K8RhUisfqw3TTLHj1uhS66YX2LZPxS4oaf33rp9HlfqSBeP
# ejlYeEdU740GKQM7SaVSH3TbBL8R6HwX9QVpGnXPlKdE4fBIn5BBFnV+KwPxRNUN
# K6lYk2y1WSKour4hJN0SMkoaNV8hyyADiX1xuTxKaXN12HgR+8WulU2d6zhzXomJ
# 2PleI9V2yfmfXSPGYanGgxzqI+ShoOGLomMd3mJt92nm7Mheng/TBeSA2z4I78Jp
# wGpTRHiT7yHqBiV2ngUIyCtd0pZ8zg3S7bk4QC4RrcnKJ3FbjyPAGogmoiZ33c1H
# G93Vp6lJ415ERcC7bFQMRbxqrMVANiav1k425zYyFMyLNyE1QulQSgDpW9rtvVcI
# H7WvG9sqYup9j8z9J1XqbBZPJ5XLln8mS8wWmdDLnBHXgYly/p1DhoQo5fkCAwEA
# AaOCAYswggGHMA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB
# /wQMMAoGCCsGAQUFBwMIMCAGA1UdIAQZMBcwCAYGZ4EMAQQCMAsGCWCGSAGG/WwH
# ATAfBgNVHSMEGDAWgBS6FtltTYUvcyl2mi91jGogj57IbzAdBgNVHQ4EFgQUn1cs
# A3cOKBWQZqVjXu5Pkh92oFswWgYDVR0fBFMwUTBPoE2gS4ZJaHR0cDovL2NybDMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0UlNBNDA5NlNIQTI1NlRpbWVT
# dGFtcGluZ0NBLmNybDCBkAYIKwYBBQUHAQEEgYMwgYAwJAYIKwYBBQUHMAGGGGh0
# dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBYBggrBgEFBQcwAoZMaHR0cDovL2NhY2Vy
# dHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0UlNBNDA5NlNIQTI1NlRp
# bWVTdGFtcGluZ0NBLmNydDANBgkqhkiG9w0BAQsFAAOCAgEAPa0eH3aZW+M4hBJH
# 2UOR9hHbm04IHdEoT8/T3HuBSyZeq3jSi5GXeWP7xCKhVireKCnCs+8GZl2uVYFv
# Qe+pPTScVJeCZSsMo1JCoZN2mMew/L4tpqVNbSpWO9QGFwfMEy60HofN6V51sMLM
# XNTLfhVqs+e8haupWiArSozyAmGH/6oMQAh078qRh6wvJNU6gnh5OruCP1QUAvVS
# u4kqVOcJVozZR5RRb/zPd++PGE3qF1P3xWvYViUJLsxtvge/mzA75oBfFZSbdakH
# Je2BVDGIGVNVjOp8sNt70+kEoMF+T6tptMUNlehSR7vM+C13v9+9ZOUKzfRUAYSy
# yEmYtsnpltD/GWX8eM70ls1V6QG/ZOB6b6Yum1HvIiulqJ1Elesj5TMHq8CWT/xr
# W7twipXTJ5/i5pkU5E16RSBAdOp12aw8IQhhA/vEbFkEiF2abhuFixUDobZaA0Vh
# qAsMHOmaT3XThZDNi5U2zHKhUs5uHHdG6BoQau75KiNbh0c+hatSF+02kULkftAR
# jsyEpHKsF7u5zKRbt5oK5YGwFvgc4pEVUNytmB3BpIiowOIIuDgP5M9WArHYSAR1
# 6gc0dP2XdkMEP5eBsX7bf/MGN4K3HP50v/01ZHo/Z5lGLvNwQ7XHBx1yomzLP8lx
# 4Q1zZKDyHcp4VQJLu2kWTsKsOqQwggauMIIElqADAgECAhAHNje3JFR82Ees/Shm
# Kl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
# Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERp
# Z2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIy
# MzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7
# MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1l
# U3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDGhjUG
# SbPBPXJJUVXHJQPE8pE3qZdRodbSg9GeTKJtoLDMg/la9hGhRBVCX6SI82j6ffOc
# iQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0hNoR8XOxs+4rgISKIhjf69o9xBd/qxkr
# PkLcZ47qUT3w1lbU5ygt69OxtXXnHwZljZQp09nsad/ZkIdGAHvbREGJ3HxqV3rw
# N3mfXazL6IRktFLydkf3YYMZ3V+0VAshaG43IbtArF+y3kp9zvU5EmfvDqVjbOSm
# xR3NNg1c1eYbqMFkdECnwHLFuk4fsbVYTXn+149zk6wsOeKlSNbwsDETqVcplicu
# 9Yemj052FVUmcJgmf6AaRyBD40NjgHt1biclkJg6OBGz9vae5jtb7IHeIhTZgirH
# kr+g3uM+onP65x9abJTyUpURK1h0QCirc0PO30qhHGs4xSnzyqqWc0Jon7ZGs506
# o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklN
# iyDSLFc1eSuo80VgvCONWPfcYd6T/jnA+bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGT
# yYwMO1uKIqjBJgj5FBASA31fI7tk42PgpuE+9sJ0sj8eCXbsq11GdeJgo1gJASgA
# DoRU7s7pXcheMBK9Rp6103a50g5rmQzSM7TNsQIDAQABo4IBXTCCAVkwEgYDVR0T
# AQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUuhbZbU2FL3MpdpovdYxqII+eyG8wHwYD
# VR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMG
# A1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsGAQUFBwEBBGswaTAkBggrBgEFBQcwAYYY
# aHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAChjVodHRwOi8vY2Fj
# ZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNydDBDBgNV
# HR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRU
# cnVzdGVkUm9vdEc0LmNybDAgBgNVHSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1s
# BwEwDQYJKoZIhvcNAQELBQADggIBAH1ZjsCTtm+YqUQiAX5m1tghQuGwGC4QTRPP
# MFPOvxj7x1Bd4ksp+3CKDaopafxpwc8dB+k+YMjYC+VcW9dth/qEICU0MWfNthKW
# b8RQTGIdDAiCqBa9qVbPFXONASIlzpVpP0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpP
# kWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp876i8dU+6WvepELJd6f8oVInw1YpxdmXa
# zPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2nrF5mYGjVoarCkXJ38SNoOeY+/umnXKv
# xMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl6
# 3f8lY5knLD0/a6fxZsNBzU+2QJshIUDQtxMkzdwdeDrknq3lNHGS1yZr5Dhzq6YB
# T70/O3itTK37xJV77QpfMzmHQXh6OOmc4d0j/R0o08f56PGYX/sr2H7yRp11LB4n
# LCbbbxV7HhmLNriT1ObyF5lZynDwN7+YAN8gFk8n+2BnFqFmut1VwDophrCYoCvt
# lUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZvAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm
# 2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQrH4D6wPIOK+XW+6kvRBVK5xMOHds3OBqh
# K/bt1nz8MIIFjTCCBHWgAwIBAgIQDpsYjvnQLefv21DiCEAYWjANBgkqhkiG9w0B
# AQwFADBlMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBBc3N1cmVk
# IElEIFJvb3QgQ0EwHhcNMjIwODAxMDAwMDAwWhcNMzExMTA5MjM1OTU5WjBiMQsw
# CQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cu
# ZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQw
# ggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC/5pBzaN675F1KPDAiMGkz
# 7MKnJS7JIT3yithZwuEppz1Yq3aaza57G4QNxDAf8xukOBbrVsaXbR2rsnnyyhHS
# 5F/WBTxSD1Ifxp4VpX6+n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQLt+C8weE5nQ7
# bXHiLQwb7iDVySAdYyktzuxeTsiT+CFhmzTrBcZe7FsavOvJz82sNEBfsXpm7nfI
# SKhmV1efVFiODCu3T6cw2Vbuyntd463JT17lNecxy9qTXtyOj4DatpGYQJB5w3jH
# trHEtWoYOAMQjdjUN6QuBX2I9YI+EJFwq1WCQTLX2wRzKm6RAXwhTNS8rhsDdV14
# Ztk6MUSaM0C/CNdaSaTC5qmgZ92kJ7yhTzm1EVgX9yRcRo9k98FpiHaYdj1ZXUJ2
# h4mXaXpI8OCiEhtmmnTK3kse5w5jrubU75KSOp493ADkRSWJtppEGSt+wJS00mFt
# 6zPZxd9LBADMfRyVw4/3IbKyEbe7f/LVjHAsQWCqsWMYRJUadmJ+9oCw++hkpjPR
# iQfhvbfmQ6QYuKZ3AeEPlAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56rF+NP8m800ER
# ElvlEFDrMcXKchYiCd98THU/Y+whX8QgUWtvsauGi0/C1kVfnSD8oR7FwI+isX4K
# Jpn15GkvmB0t9dmpsh3lGwIDAQABo4IBOjCCATYwDwYDVR0TAQH/BAUwAwEB/zAd
# BgNVHQ4EFgQU7NfjgtJxXWRM3y5nP+e6mK4cD08wHwYDVR0jBBgwFoAUReuir/SS
# y4IxLVGLp6chnfNtyA8wDgYDVR0PAQH/BAQDAgGGMHkGCCsGAQUFBwEBBG0wazAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAC
# hjdodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURS
# b290Q0EuY3J0MEUGA1UdHwQ+MDwwOqA4oDaGNGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwEQYDVR0gBAowCDAGBgRV
# HSAAMA0GCSqGSIb3DQEBDAUAA4IBAQBwoL9DXFXnOF+go3QbPbYW1/e/Vwe9mqyh
# hyzshV6pGrsi+IcaaVQi7aSId229GhT0E0p6Ly23OO/0/4C5+KH38nLeJLxSA8hO
# 0Cre+i1Wz/n096wwepqLsl7Uz9FDRJtDIeuWcqFItJnLnU+nBgMTdydE1Od/6Fmo
# 8L8vC6bp8jQ87PcDx4eo0kxAGTVGamlUsLihVo7spNU96LHc/RzY9HdaXFSMb++h
# UD38dglohJ9vytsgjTVgHAIDyyCwrFigDkBjxZgiwbJZ9VVrzyerbHbObyMt9H5x
# aiNrIv8SuFQtJ37YOtnwtoeW/VvRXKwYw02fc7cBqZ9Xql4o4rmUMYIDdjCCA3IC
# AQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xOzA5
# BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0
# YW1waW5nIENBAhALrma8Wrp/lYfG+ekE4zMEMA0GCWCGSAFlAwQCAQUAoIHRMBoG
# CSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcNMjUwNDI4
# MTQ0NzQ1WjArBgsqhkiG9w0BCRACDDEcMBowGDAWBBTb04XuYtvSPnvk9nFIUIck
# 1YZbRTAvBgkqhkiG9w0BCQQxIgQgbjdgI0b5ayqo4A8lz5gsP0sexlLx6Xvyt+Oq
# hfcuekEwNwYLKoZIhvcNAQkQAi8xKDAmMCQwIgQgdnafqPJjLx9DCzojMK7WVnX+
# 13PbBdZluQWTmEOPmtswDQYJKoZIhvcNAQEBBQAEggIAu6PhKA5jj5H2/73Ek1tT
# FEiVh063/TCaE0CnHYn5SsVR97zKSAUo3t7dXBEBb0PPjgaH7SsE8PbHmOmD6d4V
# k7Jds8sUBel+FaeM3FUXe3Kqdjb0aR4pbhxW5wm0GNB41YDOGuT2EddO5Tt9sCwA
# 7HoXrneoEBbfNkz7r++CzTN6BMDpOKlokFnP8SRXRipre3FiUngj9h/NbAaKJ6+b
# T95348U9SSPrKJrb/c2mcxy8u0bdK860bB4CHU4sDodb66eqKtaGS9x0vCFF525B
# BDxJqtJq0uTRHSIeHelbq0Tsw69NX8Hn4sl3mFxmsxFXzKz1H7UDaf9pNn4RAFGV
# ZALy9UvPYRFt3W3CUoH3EheRri39W+Mjtok4tRnRmVburLEIlp2tcjSff0PWLELJ
# kMhktcL+r/lL62FWw674+HTCFxXAwFce5ZuaNG5irAYG7/uf0/H+chleHCRvPO+n
# cgsCqYmOBKr2IhtJYsZCCei7KlCqnZVX4PHIz4mQZFWOO9RK5rSHrMbyFrRjhT54
# FFetB9AqNB+ruse4l5IMcVl6G9GTkROX3l6XHTI0PxrHNJow1t+RUIsNq3OWfUks
# B42pwA0yV2RH6qXE574mrjcj8xB/MCZGun7gCQdR7fe7QE1kYlOejTsdKL7N9El/
# 6XarYBEEbYVvefI3We1FfoY=
# SIG # End signature block
